<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\StorePackageRequest;
use App\Http\Requests\UpdatePackageRequest;
use App\Models\Category;
use App\Models\Destination;
use App\Models\Package;
use Illuminate\Support\Str;

class PackageController extends Controller
{
    public $role;

    public function __construct()
    {
        $this->role = strtolower(user()->roles()->whereIn('name', ['admin', 'agent'])->first()->name);
        view()->share('role', $this->role);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $package = Package::query();
        if (user()->hasRole('agent')) {
            $package->where('agent_id', user()->id);
        }
        $packages = $package->with(['category:id,name', 'destination:id,name'])->latest()->get();

        return view('admin.package.index', compact('packages'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $package = new Package;
        $categories = Category::getType('package')->select('name', 'id')->get();
        $destinations = Destination::where('status', true)->select('name', 'id')->get();

        return view('admin.package.create', compact('package', 'categories', 'destinations'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StorePackageRequest $request)
    {
        $data = $request->validated();
        $data['status'] = $request->has('status');
        $data['slug'] = Str::slug($data['name']);

        if (user()->hasRole('agent')) {
            $data['agent_id'] = user()->id;
        }

        $data['published'] = $data['featured'] = false;
        if (user()->hasRole('admin')) {
            $data['published'] = $request->has('published');
            $data['featured'] = $request->has('featured');
            $data['published_at'] = now();
        }

        if (Package::where('slug', $data['slug'])->where('destination_id', $data['destination_id'])->where('category_id', $data['category_id'])->exists()) {
            return redirect()->back()->with('error', 'Package with this name already exists.');
        }

        try {
            $package = Package::updateOrCreate([
                'slug' => $data['slug'],
            ], $data);
            if ($request->has('thumbnail')) {
                $package->images()->create([
                    'path' => $request->file('thumbnail')->store('packages', 'public'),
                    'type' => 'thumbnail',
                ]);
            }
            if ($request->has('gallery')) {
                foreach ($request->file('gallery') as $image) {
                    $package->images()->create([
                        'path' => $image->store('packages', 'public'),
                        'type' => 'gallery',
                    ]);
                }
            }

            return redirect()->route($this->role.'.package.index')->with('success', 'Package created successfully.');
        } catch (\Exception $e) {
            // return $e->getMessage();
            return redirect()->back()->with('error', 'Package could not be created.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Package $package)
    {
        abort(404);

        return view('admin.package.show', compact('package'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Package $package)
    {
        $categories = Category::getType('package')->select('name', 'id')->get();
        $destinations = Destination::where('status', true)->select('name', 'id')->get();

        return view('admin.package.create', compact('package', 'categories', 'destinations'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdatePackageRequest $request, Package $package)
    {
        $data = $request->validated();
        $data['status'] = $request->has('status');
        $data['slug'] = Str::slug($data['name']);
        $data['inclusions'] = $request->has('inclusions') ? $data['inclusions'] : [];
        $data['exclusions'] = $request->has('exclusions') ? $data['exclusions'] : [];
        $data['we_expect'] = $request->has('we_expect') ? $data['we_expect'] : [];

        $data['published'] = $data['featured'] = false;
        if (user()->hasRole('admin')) {
            $data['published'] = $request->has('published');
            $data['featured'] = $request->has('featured');
            $data['published_at'] = now();
        }

        if (Package::where('slug', $data['slug'])->where('destination_id', $data['destination_id'])->where('category_id', $data['category_id'])->where('id', '!=', $package->id)->exists()) {
            return redirect()->back()->with('error', 'Package with this name already exists.');
        }

        try {
            $package->update($data);
            if ($request->has('thumbnail')) {
                $package->images()->where('type', 'thumbnail')->delete();
                $package->images()->create([
                    'path' => $request->file('thumbnail')->store('packages', 'public'),
                    'type' => 'thumbnail',
                ]);
            }
            if ($request->has('gallery')) {
                $package->images()->where('type', 'gallery')->delete();
                foreach ($request->file('gallery') as $image) {
                    $package->images()->create([
                        'path' => $image->store('packages', 'public'),
                        'type' => 'gallery',
                    ]);
                }
            }

            return redirect()->route($this->role.'.package.index')->with('success', 'Package updated successfully.');
        } catch (\Exception $e) {
            // return $e->getMessage();
            return redirect()->back()->with('error', 'Package could not be updated.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Package $package)
    {
        try {
            $path = public_path('storage/'.$package->thumbnail);
            $package->delete();
            if (file_exists($path)) {
                @unlink($path);
            }

            return response()->json(['message' => 'Package deleted successfully.', 'status' => 'success']);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Package could not be deleted.', 'status' => 'error'], 500);
        }
    }
}
